/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.client.transport.uasc;

import com.google.common.collect.Maps;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.CompositeByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.ByteToMessageCodec;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.Timeout;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.milo.opcua.stack.client.UaStackClientConfig;
import org.eclipse.milo.opcua.stack.client.security.ClientCertificateValidator;
import org.eclipse.milo.opcua.stack.client.transport.UaTransportRequest;
import org.eclipse.milo.opcua.stack.client.transport.uasc.ClientSecureChannel;
import org.eclipse.milo.opcua.stack.client.transport.uasc.UascClientAcknowledgeHandler;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.UaSerializationException;
import org.eclipse.milo.opcua.stack.core.UaServiceFaultException;
import org.eclipse.milo.opcua.stack.core.channel.ChannelSecurity;
import org.eclipse.milo.opcua.stack.core.channel.ChunkDecoder;
import org.eclipse.milo.opcua.stack.core.channel.ChunkEncoder;
import org.eclipse.milo.opcua.stack.core.channel.MessageAbortException;
import org.eclipse.milo.opcua.stack.core.channel.MessageDecodeException;
import org.eclipse.milo.opcua.stack.core.channel.MessageEncodeException;
import org.eclipse.milo.opcua.stack.core.channel.SerializationQueue;
import org.eclipse.milo.opcua.stack.core.channel.headers.AsymmetricSecurityHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.HeaderDecoder;
import org.eclipse.milo.opcua.stack.core.channel.messages.ErrorMessage;
import org.eclipse.milo.opcua.stack.core.channel.messages.MessageType;
import org.eclipse.milo.opcua.stack.core.channel.messages.TcpMessageDecoder;
import org.eclipse.milo.opcua.stack.core.security.SecurityPolicy;
import org.eclipse.milo.opcua.stack.core.serialization.UaResponseMessage;
import org.eclipse.milo.opcua.stack.core.types.builtin.ByteString;
import org.eclipse.milo.opcua.stack.core.types.builtin.DateTime;
import org.eclipse.milo.opcua.stack.core.types.builtin.StatusCode;
import org.eclipse.milo.opcua.stack.core.types.builtin.unsigned.Unsigned;
import org.eclipse.milo.opcua.stack.core.types.enumerated.SecurityTokenRequestType;
import org.eclipse.milo.opcua.stack.core.types.structured.ChannelSecurityToken;
import org.eclipse.milo.opcua.stack.core.types.structured.CloseSecureChannelRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.OpenSecureChannelRequest;
import org.eclipse.milo.opcua.stack.core.types.structured.OpenSecureChannelResponse;
import org.eclipse.milo.opcua.stack.core.types.structured.RequestHeader;
import org.eclipse.milo.opcua.stack.core.types.structured.ServiceFault;
import org.eclipse.milo.opcua.stack.core.util.BufferUtil;
import org.eclipse.milo.opcua.stack.core.util.CertificateUtil;
import org.eclipse.milo.opcua.stack.core.util.LongSequence;
import org.eclipse.milo.opcua.stack.core.util.NonceUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UascClientMessageHandler
extends ByteToMessageCodec<UaTransportRequest>
implements HeaderDecoder {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private List<ByteBuf> chunkBuffers = new ArrayList<ByteBuf>();
    private final AtomicReference<AsymmetricSecurityHeader> headerRef = new AtomicReference();
    private final Map<Long, UaTransportRequest> pending = Maps.newConcurrentMap();
    private final LongSequence requestIdSequence = new LongSequence(1L, 0xFFFFFFFFL);
    private ScheduledFuture renewFuture;
    private Timeout secureChannelTimeout;
    private final int maxChunkCount;
    private final int maxChunkSize;
    private final UaStackClientConfig config;
    private final ClientSecureChannel secureChannel;
    private final SerializationQueue serializationQueue;
    private final CompletableFuture<ClientSecureChannel> handshakeFuture;

    UascClientMessageHandler(UaStackClientConfig config, ClientSecureChannel secureChannel, SerializationQueue serializationQueue, CompletableFuture<ClientSecureChannel> handshakeFuture) {
        this.config = config;
        this.secureChannel = secureChannel;
        this.serializationQueue = serializationQueue;
        this.handshakeFuture = handshakeFuture;
        handshakeFuture.thenAccept(sc -> {
            Channel channel = sc.getChannel();
            channel.eventLoop().execute(() -> {
                List<UaTransportRequest> awaitingHandshake = channel.attr(UascClientAcknowledgeHandler.KEY_AWAITING_HANDSHAKE).get();
                if (awaitingHandshake != null) {
                    channel.attr(UascClientAcknowledgeHandler.KEY_AWAITING_HANDSHAKE).set(null);
                    this.logger.debug("{} message(s) queued before handshake completed; sending now.", (Object)awaitingHandshake.size());
                    awaitingHandshake.forEach(channel::writeAndFlush);
                }
            });
        });
        this.maxChunkCount = serializationQueue.getParameters().getLocalMaxChunkCount();
        this.maxChunkSize = serializationQueue.getParameters().getLocalReceiveBufferSize();
    }

    @Override
    public void handlerAdded(ChannelHandlerContext ctx) {
        SecurityTokenRequestType requestType = this.secureChannel.getChannelId() == 0L ? SecurityTokenRequestType.Issue : SecurityTokenRequestType.Renew;
        this.secureChannelTimeout = this.config.getWheelTimer().newTimeout(timeout -> {
            if (!timeout.isCancelled()) {
                this.handshakeFuture.completeExceptionally(new UaException(0x800A0000L, "timed out waiting for secure channel"));
                ctx.close();
            }
        }, this.config.getRequestTimeout().longValue(), TimeUnit.MILLISECONDS);
        this.logger.debug("OpenSecureChannel timeout scheduled for +{}ms", (Object)this.config.getRequestTimeout());
        this.sendOpenSecureChannelRequest(ctx, requestType);
    }

    @Override
    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        if (this.renewFuture != null) {
            this.renewFuture.cancel(false);
        }
        UaException exception = new UaException(2158886912L, "connection closed");
        this.handshakeFuture.completeExceptionally(exception);
        this.pending.values().forEach(request -> request.getFuture().completeExceptionally(exception));
        super.channelInactive(ctx);
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        this.logger.error("[remote={}] Exception caught: {}", ctx.channel().remoteAddress(), cause.getMessage(), cause);
        this.chunkBuffers.forEach(ReferenceCountUtil::safeRelease);
        this.chunkBuffers.clear();
        this.handshakeFuture.completeExceptionally(cause);
        ctx.close();
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) {
        if (evt instanceof CloseSecureChannelRequest) {
            this.sendCloseSecureChannelRequest(ctx, (CloseSecureChannelRequest)evt);
        }
    }

    private void sendOpenSecureChannelRequest(ChannelHandlerContext ctx, SecurityTokenRequestType requestType) {
        ByteString clientNonce = this.secureChannel.isSymmetricSigningEnabled() ? NonceUtil.generateNonce(this.secureChannel.getSecurityPolicy()) : ByteString.NULL_VALUE;
        this.secureChannel.setLocalNonce(clientNonce);
        RequestHeader header = new RequestHeader(null, DateTime.now(), Unsigned.uint(0), Unsigned.uint(0), null, this.config.getRequestTimeout(), null);
        OpenSecureChannelRequest request = new OpenSecureChannelRequest(header, Unsigned.uint(0L), requestType, this.secureChannel.getMessageSecurityMode(), this.secureChannel.getLocalNonce(), this.config.getChannelLifetime());
        this.serializationQueue.encode((binaryEncoder, chunkEncoder) -> {
            ByteBuf messageBuffer = BufferUtil.pooledBuffer();
            try {
                binaryEncoder.setBuffer(messageBuffer);
                binaryEncoder.writeMessage(null, request);
                this.checkMessageSize(messageBuffer);
                ChunkEncoder.EncodedMessage encodedMessage = chunkEncoder.encodeAsymmetric(this.secureChannel, this.requestIdSequence.getAndIncrement(), messageBuffer, MessageType.OpenSecureChannel);
                CompositeByteBuf chunkComposite = BufferUtil.compositeBuffer();
                for (ByteBuf chunk : encodedMessage.getMessageChunks()) {
                    chunkComposite.addComponent(chunk);
                    chunkComposite.writerIndex(chunkComposite.writerIndex() + chunk.readableBytes());
                }
                ctx.writeAndFlush(chunkComposite, ctx.voidPromise());
                ChannelSecurity channelSecurity = this.secureChannel.getChannelSecurity();
                long currentTokenId = -1L;
                if (channelSecurity != null) {
                    currentTokenId = channelSecurity.getCurrentToken().getTokenId().longValue();
                }
                long previousTokenId = -1L;
                if (channelSecurity != null) {
                    previousTokenId = channelSecurity.getPreviousToken().map(token -> token.getTokenId().longValue()).orElse(-1L);
                }
                this.logger.debug("Sent OpenSecureChannelRequest ({}, id={}, currentToken={}, previousToken={}).", request.getRequestType(), this.secureChannel.getChannelId(), currentTokenId, previousTokenId);
            }
            catch (MessageEncodeException e) {
                this.logger.error("Error encoding {}: {}", request, e.getMessage(), e);
                ctx.close();
            }
            finally {
                messageBuffer.release();
            }
        });
    }

    private void checkMessageSize(ByteBuf messageBuffer) throws UaSerializationException {
        int messageSize = messageBuffer.readableBytes();
        int remoteMaxMessageSize = this.serializationQueue.getParameters().getRemoteMaxMessageSize();
        if (remoteMaxMessageSize > 0 && messageSize > remoteMaxMessageSize) {
            throw new UaSerializationException(0x80B80000L, "request exceeds remote max message size: " + messageSize + " > " + remoteMaxMessageSize);
        }
    }

    private void sendCloseSecureChannelRequest(ChannelHandlerContext ctx, CloseSecureChannelRequest request) {
        this.serializationQueue.encode((binaryEncoder, chunkEncoder) -> {
            ByteBuf messageBuffer = BufferUtil.pooledBuffer();
            try {
                binaryEncoder.setBuffer(messageBuffer);
                binaryEncoder.writeMessage(null, request);
                this.checkMessageSize(messageBuffer);
                ChunkEncoder.EncodedMessage encodedMessage = chunkEncoder.encodeSymmetric(this.secureChannel, this.requestIdSequence.getAndIncrement(), messageBuffer, MessageType.CloseSecureChannel);
                CompositeByteBuf chunkComposite = BufferUtil.compositeBuffer();
                for (ByteBuf chunk : encodedMessage.getMessageChunks()) {
                    chunkComposite.addComponent(chunk);
                    chunkComposite.writerIndex(chunkComposite.writerIndex() + chunk.readableBytes());
                }
                ctx.writeAndFlush(chunkComposite).addListener((GenericFutureListener<? extends Future<? super Void>>)((GenericFutureListener<Future>)future -> ctx.close()));
                this.secureChannel.setChannelId(0L);
            }
            catch (MessageEncodeException e) {
                this.logger.error("Error encoding {}: {}", request, e.getMessage(), e);
                this.handshakeFuture.completeExceptionally(e);
                ctx.close();
            }
            catch (UaSerializationException e) {
                this.logger.error("Error serializing {}: {}", request, e.getMessage(), e);
                this.handshakeFuture.completeExceptionally(e);
                ctx.close();
            }
            finally {
                messageBuffer.release();
            }
        });
    }

    @Override
    protected void encode(ChannelHandlerContext ctx, UaTransportRequest request, ByteBuf buffer) {
        this.serializationQueue.encode((binaryEncoder, chunkEncoder) -> {
            ByteBuf messageBuffer = BufferUtil.pooledBuffer();
            try {
                binaryEncoder.setBuffer(messageBuffer);
                binaryEncoder.writeMessage(null, request.getRequest());
                this.checkMessageSize(messageBuffer);
                ChunkEncoder.EncodedMessage encodedMessage = chunkEncoder.encodeSymmetric(this.secureChannel, this.requestIdSequence.getAndIncrement(), messageBuffer, MessageType.SecureMessage);
                long requestId = encodedMessage.getRequestId();
                List<ByteBuf> messageChunks = encodedMessage.getMessageChunks();
                this.pending.put(requestId, request);
                request.getFuture().whenComplete((r, x) -> this.pending.remove(requestId));
                CompositeByteBuf chunkComposite = BufferUtil.compositeBuffer();
                for (ByteBuf chunk : messageChunks) {
                    chunkComposite.addComponent(chunk);
                    chunkComposite.writerIndex(chunkComposite.writerIndex() + chunk.readableBytes());
                }
                ctx.writeAndFlush(chunkComposite, ctx.voidPromise());
            }
            catch (MessageEncodeException e) {
                this.logger.error("Error encoding {}: {}", request.getRequest(), e.getMessage(), e);
                request.getFuture().completeExceptionally(e);
                ctx.close();
            }
            catch (UaSerializationException e) {
                this.logger.error("Error serializing {}: {}", request.getRequest(), e.getMessage(), e);
                request.getFuture().completeExceptionally(e);
            }
            finally {
                messageBuffer.release();
            }
        });
    }

    @Override
    protected void decode(ChannelHandlerContext ctx, ByteBuf buffer, List<Object> out) throws Exception {
        if (buffer.readableBytes() >= 8) {
            int messageLength = this.getMessageLength(buffer, this.maxChunkSize);
            if (buffer.readableBytes() >= messageLength) {
                this.decodeMessage(ctx, buffer, messageLength);
            }
        }
    }

    private void decodeMessage(ChannelHandlerContext ctx, ByteBuf buffer, int messageLength) throws UaException {
        MessageType messageType = MessageType.fromMediumInt(buffer.getMediumLE(buffer.readerIndex()));
        switch (messageType) {
            case OpenSecureChannel: {
                this.onOpenSecureChannel(ctx, buffer.readSlice(messageLength));
                break;
            }
            case SecureMessage: {
                this.onSecureMessage(ctx, buffer.readSlice(messageLength));
                break;
            }
            case Error: {
                this.onError(ctx, buffer.readSlice(messageLength));
                break;
            }
            default: {
                throw new UaException(2155741184L, "unexpected MessageType: " + (Object)((Object)messageType));
            }
        }
    }

    private boolean accumulateChunk(ByteBuf buffer) throws UaException {
        int chunkSize = buffer.readerIndex(0).readableBytes();
        if (chunkSize > this.maxChunkSize) {
            throw new UaException(0x80800000L, String.format("max chunk size exceeded (%s)", this.maxChunkSize));
        }
        this.chunkBuffers.add(buffer.retain());
        if (this.maxChunkCount > 0 && this.chunkBuffers.size() > this.maxChunkCount) {
            throw new UaException(0x80800000L, String.format("max chunk count exceeded (%s)", this.maxChunkCount));
        }
        char chunkType = (char)buffer.getByte(3);
        return chunkType == 'A' || chunkType == 'F';
    }

    private void onOpenSecureChannel(ChannelHandlerContext ctx, ByteBuf buffer) throws UaException {
        if (this.secureChannelTimeout != null) {
            if (this.secureChannelTimeout.cancel()) {
                this.logger.debug("OpenSecureChannel timeout canceled");
                this.secureChannelTimeout = null;
            } else {
                this.logger.warn("timed out waiting for secure channel");
                this.handshakeFuture.completeExceptionally(new UaException(0x800A0000L, "timed out waiting for secure channel"));
                ctx.close();
                return;
            }
        }
        buffer.skipBytes(12);
        AsymmetricSecurityHeader securityHeader = AsymmetricSecurityHeader.decode(buffer, this.config.getEncodingLimits());
        if (this.headerRef.compareAndSet(null, securityHeader)) {
            ClientCertificateValidator certificateValidator = this.config.getCertificateValidator();
            SecurityPolicy securityPolicy = SecurityPolicy.fromUri(securityHeader.getSecurityPolicyUri());
            if (securityPolicy != SecurityPolicy.None) {
                ByteString serverCertificateBytes = securityHeader.getSenderCertificate();
                List<X509Certificate> serverCertificateChain = CertificateUtil.decodeCertificates(serverCertificateBytes.bytesOrEmpty());
                certificateValidator.validateCertificateChain(serverCertificateChain);
            }
        } else if (!securityHeader.equals(this.headerRef.get())) {
            throw new UaException(2148728832L, "subsequent AsymmetricSecurityHeader did not match");
        }
        if (this.accumulateChunk(buffer)) {
            List<ByteBuf> buffersToDecode = this.chunkBuffers;
            this.chunkBuffers = new ArrayList<ByteBuf>(this.maxChunkCount);
            this.serializationQueue.decode((binaryDecoder, chunkDecoder) -> {
                ByteBuf message;
                try {
                    ChunkDecoder.DecodedMessage decodedMessage = chunkDecoder.decodeAsymmetric(this.secureChannel, buffersToDecode);
                    message = decodedMessage.getMessage();
                }
                catch (MessageAbortException e) {
                    this.logger.warn("Received message abort chunk; error={}, reason={}", (Object)e.getStatusCode(), (Object)e.getMessage());
                    return;
                }
                catch (MessageDecodeException e) {
                    this.logger.error("Error decoding asymmetric message", e);
                    this.handshakeFuture.completeExceptionally(e);
                    ctx.close();
                    return;
                }
                try {
                    UaResponseMessage response = (UaResponseMessage)binaryDecoder.setBuffer(message).readMessage(null);
                    StatusCode serviceResult = response.getResponseHeader().getServiceResult();
                    if (serviceResult.isGood()) {
                        OpenSecureChannelResponse oscr = (OpenSecureChannelResponse)response;
                        this.secureChannel.setChannelId(oscr.getSecurityToken().getChannelId().longValue());
                        this.logger.debug("Received OpenSecureChannelResponse.");
                        this.installSecurityToken(ctx, oscr);
                        this.handshakeFuture.complete(this.secureChannel);
                    } else {
                        ServiceFault serviceFault = response instanceof ServiceFault ? (ServiceFault)response : new ServiceFault(response.getResponseHeader());
                        this.handshakeFuture.completeExceptionally(new UaServiceFaultException(serviceFault));
                        ctx.close();
                    }
                }
                catch (Throwable t) {
                    this.logger.error("Error decoding OpenSecureChannelResponse", t);
                    this.handshakeFuture.completeExceptionally(t);
                    ctx.close();
                }
                finally {
                    message.release();
                }
            });
        }
    }

    private void installSecurityToken(ChannelHandlerContext ctx, OpenSecureChannelResponse response) throws UaException {
        ChannelSecurity oldSecrets;
        if (response.getServerProtocolVersion().longValue() < 0L) {
            throw new UaException(2159935488L, "server protocol version unsupported: " + response.getServerProtocolVersion());
        }
        ChannelSecurity.SecurityKeys newKeys = null;
        ChannelSecurityToken newToken = response.getSecurityToken();
        if (this.secureChannel.isSymmetricSigningEnabled()) {
            ByteString serverNonce = response.getServerNonce();
            NonceUtil.validateNonce(serverNonce, this.secureChannel.getSecurityPolicy());
            this.secureChannel.setRemoteNonce(serverNonce);
            newKeys = ChannelSecurity.generateKeyPair(this.secureChannel, this.secureChannel.getLocalNonce(), this.secureChannel.getRemoteNonce());
        }
        ChannelSecurity.SecurityKeys oldKeys = (oldSecrets = this.secureChannel.getChannelSecurity()) != null ? oldSecrets.getCurrentKeys() : null;
        ChannelSecurityToken oldToken = oldSecrets != null ? oldSecrets.getCurrentToken() : null;
        this.secureChannel.setChannelSecurity(new ChannelSecurity(newKeys, newToken, oldKeys, oldToken));
        DateTime createdAt = response.getSecurityToken().getCreatedAt();
        long revisedLifetime = response.getSecurityToken().getRevisedLifetime().longValue();
        if (revisedLifetime > 0L) {
            long renewAt = (long)((double)revisedLifetime * 0.75);
            this.renewFuture = ctx.executor().schedule(() -> this.sendOpenSecureChannelRequest(ctx, SecurityTokenRequestType.Renew), renewAt, TimeUnit.MILLISECONDS);
        } else {
            this.logger.warn("Server revised secure channel lifetime to 0; renewal will not occur.");
        }
        ctx.executor().execute(() -> {
            if (ctx.pipeline().get(UascClientAcknowledgeHandler.class) != null) {
                ctx.pipeline().remove(UascClientAcknowledgeHandler.class);
            }
        });
        ChannelSecurity channelSecurity = this.secureChannel.getChannelSecurity();
        long currentTokenId = channelSecurity.getCurrentToken().getTokenId().longValue();
        long previousTokenId = channelSecurity.getPreviousToken().map(t -> t.getTokenId().longValue()).orElse(-1L);
        this.logger.debug("SecureChannel id={}, currentTokenId={}, previousTokenId={}, lifetime={}ms, createdAt={}", this.secureChannel.getChannelId(), currentTokenId, previousTokenId, revisedLifetime, createdAt);
    }

    private void onSecureMessage(ChannelHandlerContext ctx, ByteBuf buffer) throws UaException {
        buffer.skipBytes(8);
        long secureChannelId = buffer.readUnsignedIntLE();
        if (secureChannelId != this.secureChannel.getChannelId()) {
            throw new UaException(0x80220000L, "invalid secure channel id: " + secureChannelId);
        }
        if (this.accumulateChunk(buffer)) {
            List<ByteBuf> buffersToDecode = this.chunkBuffers;
            this.chunkBuffers = new ArrayList<ByteBuf>(this.maxChunkCount);
            this.serializationQueue.decode((binaryDecoder, chunkDecoder) -> {
                long requestId;
                ByteBuf message;
                try {
                    ChunkDecoder.DecodedMessage decodedMessage = chunkDecoder.decodeSymmetric(this.secureChannel, buffersToDecode);
                    message = decodedMessage.getMessage();
                    requestId = decodedMessage.getRequestId();
                }
                catch (MessageAbortException e) {
                    this.logger.warn("Received message abort chunk; error={}, reason={}", (Object)e.getStatusCode(), (Object)e.getMessage());
                    UaTransportRequest request = this.pending.remove(e.getRequestId());
                    if (request != null) {
                        request.getFuture().completeExceptionally(e);
                    } else {
                        this.logger.warn("No pending request for requestId={}", (Object)e.getRequestId());
                    }
                    return;
                }
                catch (MessageDecodeException e) {
                    this.logger.error("Error decoding symmetric message", e);
                    ctx.close();
                    return;
                }
                UaTransportRequest request = this.pending.remove(requestId);
                try {
                    UaResponseMessage response = (UaResponseMessage)binaryDecoder.setBuffer(message).readMessage(null);
                    if (request != null) {
                        request.getFuture().complete(response);
                    } else {
                        this.logger.warn("No pending request with requestId={} for {}", (Object)requestId, (Object)response.getClass().getSimpleName());
                    }
                }
                catch (Throwable t) {
                    this.logger.error("Error decoding UaResponseMessage", t);
                    if (request != null) {
                        request.getFuture().completeExceptionally(t);
                    }
                }
                finally {
                    message.release();
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void onError(ChannelHandlerContext ctx, ByteBuf buffer) {
        try {
            ErrorMessage errorMessage = TcpMessageDecoder.decodeError(buffer);
            StatusCode statusCode = errorMessage.getError();
            this.logger.error("[remote={}] errorMessage={}", (Object)ctx.channel().remoteAddress(), (Object)errorMessage);
            this.handshakeFuture.completeExceptionally(new UaException(statusCode, errorMessage.getReason()));
        }
        catch (UaException e) {
            this.logger.error("[remote={}] An exception occurred while decoding an error message: {}", ctx.channel().remoteAddress(), e.getMessage(), e);
            this.handshakeFuture.completeExceptionally(e);
        }
        finally {
            ctx.close();
        }
    }
}

